import {createAsyncThunk, createSlice} from "@reduxjs/toolkit";
import {
    authMe_Server,
    login_Server, otpCheck_Server,
    register_Server,
    updateUserProfileInfo_Server,
} from "@/services/apiServices";

export const
    register = createAsyncThunk(
        "authentication/register",
        async (values) => {
            try {
                const response = await register_Server(values);
                return response.data;
            } catch (e) {
                console.log(e);
                throw e.response.data.message;
            }
        }
    ),
    login = createAsyncThunk(
        "authentication/login",
        async (values) => {
            try {
                const response = await login_Server(values);
                return response.data;
            } catch (e) {
                throw e.response.data.message;
            }
        }),
    otpCheck = createAsyncThunk("authentication/otpCheck", async (values) => {
        try {
            const response = await otpCheck_Server(values);
            return response.data;
        } catch (e) {
            throw e.response.data.message;
        }
    }),
    authMe = createAsyncThunk("authentication/authMe", async (token) => {
        const response = await authMe_Server(token);
        return response.data;
    }),
    updateUserProfileInformation = createAsyncThunk(
        "authentication/updateInformation",
        async (token) => {
            try {
                const response = await updateUserProfileInfo_Server(token);
                return response.data;
            } catch (e) {
                throw e.response.data.message;
            }
        }
    );

const
    localStorageData = typeof localStorage !== "undefined" && JSON.parse(localStorage.getItem("userToken")),
    saveToLocalStorage = (state) =>
        typeof localStorage !== "undefined" && localStorage.setItem("userToken", JSON.stringify(state.token)),
    clearLocalStorageData = () => typeof localStorage !== "undefined" && localStorage.removeItem("userToken"),
    authenticationSlice = createSlice({
        name: "authentication",
        initialState: {
            isLoggedIn: false,
            token: localStorageData ? localStorageData : "",
            userInfo: null,
            status: "idle",
            error: null,
        },
        reducers: {
            loginHandler: (state) => {
                if (state.token) {
                    state.isLoggedIn = true;
                }
            },
            logoutHandler: (state) => {
                state.isLoggedIn = false;
                state.token = "";
                state.userInfo = "";
                state.status = "idle";
                clearLocalStorageData();
            },
            authStatusIdealHandler: (state) => {
                state.status = "idle";
            },
            authErrorIdealHandler: (state) => {
                state.error = null;
            }
        },
        extraReducers: (builder) => {
            builder
                .addCase(register.pending, (state) => {
                    state.status = "reg-loading";
                })
                .addCase(register.fulfilled, (state) => {
                    state.status = "reg-completed";
                })
                .addCase(register.rejected, (state, action) => {
                    state.status = "reg-failed";
                    state.error = action.error.message;
                })
                .addCase(login.pending, (state) => {
                    state.status = "log-loading";
                })
                .addCase(login.fulfilled, (state) => {
                    state.status = "log-completed";
                })
                .addCase(login.rejected, (state, action) => {
                    state.status = "log-failed";
                    state.error = action.error.message;
                })
                .addCase(otpCheck.pending, (state) => {
                    state.status = "otp-loading";
                })
                .addCase(otpCheck.fulfilled, (state, action) => {
                    state.status = "otp-completed";
                    state.token = action.payload.data.token;
                    state.userInfo = action.payload.data.user;
                    saveToLocalStorage(state);
                })
                .addCase(otpCheck.rejected, (state, action) => {
                    state.status = "otp-failed";
                    state.error = action.error.message;
                })
                .addCase(authMe.pending, (state) => {
                    state.status = "auth-loading";
                })
                .addCase(authMe.fulfilled, (state, action) => {
                    state.status = "auth-completed";
                    state.isLoggedIn = true;
                    state.userInfo = action.payload.data.user;
                })
                .addCase(authMe.rejected, (state, action) => {
                    state.status = "auth-failed";
                    state.error = action.error.message;
                })
                .addCase(updateUserProfileInformation.pending, (state) => {
                    state.status = "edit-profile-loading";
                })
                .addCase(updateUserProfileInformation.fulfilled, (state) => {
                    state.status = "edit-profile-updated";
                })
                .addCase(updateUserProfileInformation.rejected, (state, action) => {
                    state.status = "edit-profile-failed";
                    state.error = action.payload.message;
                })
        },
    });

export const
    selectLoggedInStatus = state => state.authentication.isLoggedIn,
    selectToken = state => state.authentication.token,
    selectUserInfo = state => state.authentication.userInfo,
    selectAuthenticationStatus = state => state.authentication.status,
    selectAuthenticationError = state => state.authentication.error,
    {loginHandler, logoutHandler, authStatusIdealHandler, authErrorIdealHandler} = authenticationSlice.actions;

export default authenticationSlice.reducer;
